package gov.va.med.mhv.usermgmt.filter;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.service.SessionAPIService;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.annotation.Resource;
import javax.faces.context.FacesContext;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.jaxrs.ext.MessageContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * 
 * THIS FILTER IS INTENDED TO BE ASSIGNED TO EACH SECURED API IN MHV SERVICES.
 * IT WILL CHECK THE HTTP SESSION FOR A SECURITY TOKEN AND CHECK THAT IT IS
 * STILL IN GOOD STANDING. (NOT EXPIRED AND EXISTING)
 * 
 * IF THE TOKEN IS NOT IN GOOD STANDING IT WILL AUTOMATICALLY SUBMIT A REQUEST
 * TO CREATE A NEW TOKEN AND THEN STORE IT IN THE HTTP SESSION.
 *
 */

public class ApiAddAuthSessionClientFilter implements ClientRequestFilter {
	private static Logger logger = LogManager.getLogger(ApiAddAuthSessionClientFilter.class);

	private static final String API_TOKEN_KEY = "Token";
	private static final String API_EXPIRES_KEY = "Expires";
	private static final String API_SESSION_TOKEN_KEY = "APISessionToken";
	private static final String API_SESSION_EXP_KEY = "APISessionExpiration";
	private static final String RFC1123_PATTERN = "EEE, dd MMM yyyy HH:mm:ss z";
	private static final String SESSION_USERNAME_KEY = "LIFERAY_SHARED_userid";

	@Context
	MessageContext mc;

	@Resource
	private SessionAPIService sessionServiceProxy;

	@Override
	public void filter(ClientRequestContext requestContext) throws IOException {

		PortletSession session;

		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			session = request.getPortletSession();
		} catch (NullPointerException npe) {
			IOException ioe = new IOException("Unable to access the clients http session");
			logger.error(ioe);
			throw ioe;
		}

		String token = (String) session.getAttribute(API_SESSION_TOKEN_KEY);
		String expires = (String) session.getAttribute(API_SESSION_EXP_KEY);

		Date expiresDt = null;

		if (expires != null) {
			SimpleDateFormat sdf = new SimpleDateFormat(RFC1123_PATTERN);
			try {
				expiresDt = sdf.parse(expires);
			} catch (ParseException e1) {
				logger.debug(e1);
				IOException ioe = new IOException("API Session Expiration is invalid from http session");
				logger.error(ioe);
				throw ioe;
			}
		}

		String mhvUsername = (String) session.getAttribute(SESSION_USERNAME_KEY, PortletSession.APPLICATION_SCOPE);

		boolean needsNewAPIToken = false;

		if (token == null || expiresDt == null) {
			needsNewAPIToken = true;
			logger.debug("The API Token for mhvUsername " + mhvUsername + " doesn't exist");
		}
		
		if (!needsNewAPIToken && expiresDt != null && expiresDt.before(new Date())) {
			needsNewAPIToken = true;
			logger.debug("The API Token for mhvUsername " + mhvUsername + " expired");
		}

		if (needsNewAPIToken) {
			logger.debug("Fetching a new API Token for mhvUsername " + mhvUsername);

			try {
				Response r = sessionServiceProxy.createSession();

				token = r.getHeaderString(API_TOKEN_KEY);
				expires = r.getHeaderString(API_EXPIRES_KEY);

				if (StringUtils.isEmpty(token) || StringUtils.isEmpty(expires)) {
					IOException ioe = new IOException("API Token response missing headers for 'Token' or 'Expires'");
					logger.error(ioe);
					throw ioe;
				}

				session.setAttribute(API_SESSION_TOKEN_KEY, token);
				session.setAttribute(API_SESSION_EXP_KEY, expires);
			} catch (MHVException e) {
				logger.error(e);
				throw new IOException(e);
			}
		}

		requestContext.getHeaders().add(API_TOKEN_KEY, token);
	}

}